<?php

namespace App\Http\Controllers;

use App\Models\InterviewPreparation;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;

class InterviewPreparationController extends Controller
{
    public function index(): JsonResponse
    {
        $preparations = InterviewPreparation::with(['user', 'job'])->get();
        return response()->json([
            'status' => 'success',
            'data' => $preparations,
        ], 200);
    }

    public function show($id): JsonResponse
    {
        $preparation = InterviewPreparation::with(['user', 'job'])->findOrFail($id);
        return response()->json([
            'status' => 'success',
            'data' => $preparation,
        ], 200);
    }

    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'User_id' => 'required|exists:users,User_id',
            'Tool_type' => 'required|in:mock_interview,practice_test',
            'Title' => 'required|string|max:255',
            'Content_url' => 'nullable|url|max:255',
            'Score' => 'nullable|numeric|between:0,100',
            'Is_completed' => 'required|boolean',
            'Feedback' => 'nullable|string',
            'Job_id' => 'nullable|exists:jobs,Job_id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $preparation = InterviewPreparation::create($request->only([
            'User_id',
            'Tool_type',
            'Title',
            'Content_url',
            'Score',
            'Is_completed',
            'Feedback',
            'Job_id',
        ]));

        return response()->json([
            'status' => 'success',
            'data' => $preparation->load(['user', 'job']),
        ], 201);
    }

    public function update(Request $request, $id): JsonResponse
    {
        $preparation = InterviewPreparation::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'User_id' => 'sometimes|exists:users,User_id',
            'Tool_type' => 'sometimes|in:mock_interview,practice_test',
            'Title' => 'sometimes|string|max:255',
            'Content_url' => 'nullable|url|max:255',
            'Score' => 'nullable|numeric|between:0,100',
            'Is_completed' => 'sometimes|boolean',
            'Feedback' => 'nullable|string',
            'Job_id' => 'nullable|exists:jobs,Job_id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $preparation->update($request->only([
            'User_id',
            'Tool_type',
            'Title',
            'Content_url',
            'Score',
            'Is_completed',
            'Feedback',
            'Job_id',
        ]));

        return response()->json([
            'status' => 'success',
            'data' => $preparation->load(['user', 'job']),
        ], 200);
    }

    public function destroy($id): JsonResponse
    {
        $preparation = InterviewPreparation::findOrFail($id);
        $preparation->delete();

        return response()->json([
            'status' => 'success',
            'message' => 'Interview preparation deleted successfully',
        ], 204);
    }

    // Placeholder for startPracticeTest - requires new logic without Content_id
    /*
    public function startPracticeTest($id): JsonResponse
    {
        // TODO: Implement new logic for practice test, possibly using Job_id to fetch job_questions
        return response()->json([
            'status' => 'error',
            'message' => 'Practice test functionality not implemented without Content_id',
        ], 400);
    }
    */

    // Placeholder for markVideoViewed
    public function markVideoViewed(Request $request, $id): JsonResponse
    {
        $preparation = InterviewPreparation::findOrFail($id);
        session(['video_viewed_' . $id => true]);
        return response()->json([
            'status' => 'success',
            'message' => 'Video marked as viewed.',
        ], 200);
    }

    // Placeholder for submitPracticeTest - requires new logic without Content_id
    /*
    public function submitPracticeTest(Request $request, $id): JsonResponse
    {
        // TODO: Implement new logic for practice test submission, possibly using Job_id to fetch job_questions
        return response()->json([
            'status' => 'error',
            'message' => 'Practice test submission not implemented without Content_id',
        ], 400);
    }
    */
}
